<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Model\Branch;
use App\Model\CashbackSetting;
use Brian2694\Toastr\Facades\Toastr;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Support\Facades\Validator;

class CashbackController extends Controller
{
    public function __construct(
        private CashbackSetting $cashbackSetting,
        private Branch $branch,
    ) {}

    /**
     * Display the cashback settings index
     */
    public function index(): Renderable
    {
        $globalSetting = $this->cashbackSetting->whereNull('branch_id')->first();
        $branchSettings = $this->cashbackSetting->with('branch')->whereNotNull('branch_id')->paginate(15);
        $branches = $this->branch->where('status', 1)->get();

        return view('admin-views.cashback.index', compact('globalSetting', 'branchSettings', 'branches'));
    }

    /**
     * Store a new cashback setting
     */
    public function store(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'nullable|exists:branches,id',
            'status' => 'required|boolean',
            'cashback_percentage' => 'required|numeric|min:0|max:100',
            'minimum_order_amount' => 'required|numeric|min:0',
            'maximum_cashback_amount' => 'nullable|numeric|min:0',
            'wallet_discount_status' => 'required|boolean',
            'wallet_discount_max_percentage' => 'required|numeric|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // Check if setting already exists for this branch
        $existingSetting = $this->cashbackSetting->where('branch_id', $request->branch_id)->first();
        if ($existingSetting) {
            Toastr::error(translate('Cashback setting already exists for this branch'));
            return redirect()->back();
        }

        $cashbackSetting = new CashbackSetting();
        $cashbackSetting->branch_id = $request->branch_id;
        $cashbackSetting->status = $request->status;
        $cashbackSetting->cashback_percentage = $request->cashback_percentage;
        $cashbackSetting->minimum_order_amount = $request->minimum_order_amount;
        $cashbackSetting->maximum_cashback_amount = $request->maximum_cashback_amount;
        $cashbackSetting->wallet_discount_status = $request->wallet_discount_status;
        $cashbackSetting->wallet_discount_max_percentage = $request->wallet_discount_max_percentage;
        $cashbackSetting->save();

        $branchName = $request->branch_id ? $this->branch->find($request->branch_id)->name : 'Global';
        Toastr::success(translate('Cashback setting created successfully for') . ' ' . $branchName);

        return redirect()->back();
    }

    /**
     * Show the form for editing the specified cashback setting
     */
    public function edit($id): Renderable
    {
        $cashbackSetting = $this->cashbackSetting->findOrFail($id);
        $branches = $this->branch->where('status', 1)->get();

        return view('admin-views.cashback.edit', compact('cashbackSetting', 'branches'));
    }

    /**
     * Update the specified cashback setting
     */
    public function update(Request $request, $id): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'status' => 'required|boolean',
            'cashback_percentage' => 'required|numeric|min:0|max:100',
            'minimum_order_amount' => 'required|numeric|min:0',
            'maximum_cashback_amount' => 'nullable|numeric|min:0',
            'wallet_discount_status' => 'required|boolean',
            'wallet_discount_max_percentage' => 'required|numeric|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $cashbackSetting = $this->cashbackSetting->findOrFail($id);
        $cashbackSetting->status = $request->status;
        $cashbackSetting->cashback_percentage = $request->cashback_percentage;
        $cashbackSetting->minimum_order_amount = $request->minimum_order_amount;
        $cashbackSetting->maximum_cashback_amount = $request->maximum_cashback_amount;
        $cashbackSetting->wallet_discount_status = $request->wallet_discount_status;
        $cashbackSetting->wallet_discount_max_percentage = $request->wallet_discount_max_percentage;
        $cashbackSetting->save();

        Toastr::success(translate('Cashback setting updated successfully'));
        return redirect()->route('admin.cashback.index');
    }

    /**
     * Update the status of the specified cashback setting
     */
    public function status($id, $status): RedirectResponse
    {
        $cashbackSetting = $this->cashbackSetting->findOrFail($id);
        $cashbackSetting->status = $status;
        $cashbackSetting->save();

        Toastr::success(translate('Cashback setting status updated successfully'));
        return redirect()->back();
    }

    /**
     * Remove the specified cashback setting
     */
    public function delete($id): RedirectResponse
    {
        $cashbackSetting = $this->cashbackSetting->findOrFail($id);

        // Don't allow deletion of global setting
        if ($cashbackSetting->branch_id === null) {
            Toastr::error(translate('Global cashback setting cannot be deleted'));
            return redirect()->back();
        }

        $cashbackSetting->delete();
        Toastr::success(translate('Cashback setting deleted successfully'));
        return redirect()->back();
    }

    /**
     * Get cashback settings for a specific branch
     */
    public function branchSettings($branchId): JsonResponse
    {
        $setting = CashbackSetting::getForBranch($branchId);
        return response()->json($setting);
    }
}
