<?php

namespace App\Http\Controllers\Branch;

use App\CentralLogics\Helpers;
use App\Http\Controllers\Controller;
use App\Model\Branch;
use App\Model\CashbackSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CashbackController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:branch');
    }

    public function index()
    {
        $branch = Auth::guard('branch')->user();
        $cashbackSetting = CashbackSetting::getForBranch($branch->id);

        return view('branch-views.cashback.index', compact('cashbackSetting', 'branch'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'status' => 'required|boolean',
            'cashback_percentage' => 'required|numeric|min:0|max:100',
            'minimum_order_amount' => 'required|numeric|min:0',
            'maximum_cashback_amount' => 'required|numeric|min:0',
            'wallet_discount_status' => 'required|boolean',
            'wallet_discount_max_percentage' => 'required|numeric|min:0|max:50'
        ]);

        $branch = Auth::guard('branch')->user();

        // Check if setting already exists for this branch
        $cashbackSetting = CashbackSetting::where('branch_id', $branch->id)->first();

        if ($cashbackSetting) {
            // Update existing setting
            $cashbackSetting->update([
                'status' => $request->status,
                'cashback_percentage' => $request->cashback_percentage,
                'minimum_order_amount' => $request->minimum_order_amount,
                'maximum_cashback_amount' => $request->maximum_cashback_amount,
                'wallet_discount_status' => $request->wallet_discount_status,
                'wallet_discount_max_percentage' => $request->wallet_discount_max_percentage,
            ]);
            $message = 'Cashback settings updated successfully!';
        } else {
            // Create new setting
            CashbackSetting::create([
                'branch_id' => $branch->id,
                'status' => $request->status,
                'cashback_percentage' => $request->cashback_percentage,
                'minimum_order_amount' => $request->minimum_order_amount,
                'maximum_cashback_amount' => $request->maximum_cashback_amount,
                'wallet_discount_status' => $request->wallet_discount_status,
                'wallet_discount_max_percentage' => $request->wallet_discount_max_percentage,
            ]);
            $message = 'Cashback settings created successfully!';
        }

        Helpers::add_or_update_translations(
            request: $request,
            key_data: 'cashback_updated',
            name_field: 'message',
            model_name: 'CashbackSetting',
            data_id: $cashbackSetting->id ?? null,
            data_value: $message
        );

        return redirect()->route('branch.cashback.index')->with('success', $message);
    }

    public function status(Request $request)
    {
        $branch = Auth::guard('branch')->user();
        $cashbackSetting = CashbackSetting::where('branch_id', $branch->id)->first();

        if (!$cashbackSetting) {
            return response()->json(['success' => false, 'message' => 'Cashback setting not found']);
        }

        $cashbackSetting->status = $request->status;
        $cashbackSetting->save();

        return response()->json([
            'success' => true,
            'message' => 'Cashback status updated successfully'
        ]);
    }

    public function walletDiscountStatus(Request $request)
    {
        $branch = Auth::guard('branch')->user();
        $cashbackSetting = CashbackSetting::where('branch_id', $branch->id)->first();

        if (!$cashbackSetting) {
            return response()->json(['success' => false, 'message' => 'Cashback setting not found']);
        }

        $cashbackSetting->wallet_discount_status = $request->status;
        $cashbackSetting->save();

        return response()->json([
            'success' => true,
            'message' => 'Wallet discount status updated successfully'
        ]);
    }

    public function edit()
    {
        $branch = Auth::guard('branch')->user();
        $cashbackSetting = CashbackSetting::getForBranch($branch->id);

        return view('branch-views.cashback.edit', compact('cashbackSetting', 'branch'));
    }

    public function update(Request $request)
    {
        $request->validate([
            'status' => 'required|boolean',
            'cashback_percentage' => 'required|numeric|min:0|max:100',
            'minimum_order_amount' => 'required|numeric|min:0',
            'maximum_cashback_amount' => 'required|numeric|min:0',
            'wallet_discount_status' => 'required|boolean',
            'wallet_discount_max_percentage' => 'required|numeric|min:0|max:50'
        ]);

        $branch = Auth::guard('branch')->user();
        $cashbackSetting = CashbackSetting::where('branch_id', $branch->id)->firstOrFail();

        $cashbackSetting->update([
            'status' => $request->status,
            'cashback_percentage' => $request->cashback_percentage,
            'minimum_order_amount' => $request->minimum_order_amount,
            'maximum_cashback_amount' => $request->maximum_cashback_amount,
            'wallet_discount_status' => $request->wallet_discount_status,
            'wallet_discount_max_percentage' => $request->wallet_discount_max_percentage,
        ]);

        Helpers::add_or_update_translations(
            request: $request,
            key_data: 'cashback_updated',
            name_field: 'message',
            model_name: 'CashbackSetting',
            data_id: $cashbackSetting->id,
            data_value: 'Cashback settings updated successfully!'
        );

        return redirect()->route('branch.cashback.index')->with('success', 'Cashback settings updated successfully!');
    }

    public function delete()
    {
        $branch = Auth::guard('branch')->user();
        $cashbackSetting = CashbackSetting::where('branch_id', $branch->id)->first();

        if (!$cashbackSetting) {
            return response()->json(['success' => false, 'message' => 'Cashback setting not found']);
        }

        $cashbackSetting->delete();

        return response()->json([
            'success' => true,
            'message' => 'Cashback settings deleted successfully'
        ]);
    }
}
