<?php

namespace App\Model;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class CashbackSetting extends Model
{
    protected $fillable = [
        'branch_id',
        'status',
        'cashback_percentage',
        'minimum_order_amount',
        'maximum_cashback_amount',
        'wallet_discount_status',
        'wallet_discount_max_percentage'
    ];

    protected $casts = [
        'status' => 'boolean',
        'cashback_percentage' => 'decimal:2',
        'minimum_order_amount' => 'decimal:2',
        'maximum_cashback_amount' => 'decimal:2',
        'wallet_discount_status' => 'boolean',
        'wallet_discount_max_percentage' => 'decimal:2'
    ];

    /**
     * Get the branch that owns the cashback setting
     */
    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    /**
     * Get cashback setting for a specific branch or global setting
     */
    public static function getForBranch($branchId = null)
    {
        // First try to get branch-specific setting
        if ($branchId) {
            $setting = self::where('branch_id', $branchId)->first();
            if ($setting) {
                return $setting;
            }
        }

        // Fall back to global setting (branch_id = null)
        return self::whereNull('branch_id')->first();
    }

    /**
     * Calculate cashback amount for an order
     */
    public function calculateCashback($orderAmount)
    {
        if (!$this->status || $orderAmount < $this->minimum_order_amount) {
            return 0;
        }

        $cashback = ($orderAmount * $this->cashback_percentage) / 100;

        if ($this->maximum_cashback_amount && $cashback > $this->maximum_cashback_amount) {
            $cashback = $this->maximum_cashback_amount;
        }

        return $cashback;
    }

    /**
     * Calculate maximum wallet discount allowed for an order
     */
    public function calculateMaxWalletDiscount($orderAmount)
    {
        if (!$this->wallet_discount_status) {
            return 0;
        }

        return ($orderAmount * $this->wallet_discount_max_percentage) / 100;
    }

    /**
     * Check if wallet discount is allowed
     */
    public function isWalletDiscountAllowed()
    {
        return $this->wallet_discount_status;
    }
}
